<?php
/**
 * Forumly.site - Professional Forum Script
 * Single-File PHP Forum with JSON Database
 * Version: 7.0 PRODUCTION RELEASE
 * Author: Sir. ServeIt
 * Copyright: 2025
 */

error_reporting(E_ALL);
ini_set('display_errors', 1);
session_start();

define('DATA_DIR', __DIR__ . '/data');
define('UPLOAD_DIR', __DIR__ . '/uploads');
define('VERSION', '7.0');

class Forumly {
    private $dataFiles = [];
    private $themes = [];
    private $currentUser = null;
    private $settings = null;

    public function __construct() {
        $this->initializeDataFiles();
        $this->initializeSystem();
        $this->loadSettings();
        $this->initializeThemes();
        $this->authenticateUser();
        $this->handleRouting();
    }

    private function initializeDataFiles() {
        $this->dataFiles = [
            'users' => DATA_DIR . '/users.json',
            'categories' => DATA_DIR . '/categories.json',
            'posts' => DATA_DIR . '/posts.json',
            'threads' => DATA_DIR . '/threads.json',
            'settings' => DATA_DIR . '/settings.json',
            'messages' => DATA_DIR . '/messages.json',
            'reports' => DATA_DIR . '/reports.json',
            'downloads' => DATA_DIR . '/downloads.json'
        ];
    }

    private function initializeThemes() {
        $this->themes = [
            'cyber_nexus' => 'Cyber Nexus',
            'quantum_void' => 'Quantum Void', 
            'neon_dream' => 'Neon Dream',
            'cosmic_horizon' => 'Cosmic Horizon',
            'matrix_reality' => 'Matrix Reality',
            'midnight_ocean' => 'Midnight Ocean',
            'sunset_terminal' => 'Sunset Terminal'
        ];
    }

    private function initializeSystem() {
        $dirs = [DATA_DIR, UPLOAD_DIR];
        foreach ($dirs as $dir) {
            if (!is_dir($dir)) mkdir($dir, 0755, true);
        }

        foreach ($this->dataFiles as $key => $file) {
            if (!file_exists($file)) {
                $initialData = $this->getInitialData($key);
                file_put_contents($file, json_encode($initialData, JSON_PRETTY_PRINT));
            }
        }
    }

    private function getInitialData($type) {
        switch ($type) {
            case 'users':
                return [
                    'last_id' => 1,
                    'users' => [
                        [
                            'id' => 1,
                            'username' => 'admin',
                            'email' => 'admin@yourforum.com',
                            'password_hash' => password_hash('admin123', PASSWORD_DEFAULT),
                            'role' => 'admin',
                            'joined_date' => date('Y-m-d H:i:s'),
                            'reputation' => 1000,
                            'signature' => 'Forum Administrator',
                            'avatar' => '',
                            'is_banned' => false,
                            'last_login' => null,
                            'post_count' => 0
                        ]
                    ]
                ];
                
            case 'categories':
                return [
                    'last_id' => 6,
                    'categories' => [
                        ['id' => 1, 'name' => 'General Discussion', 'description' => 'General forum discussions', 'order' => 1],
                        ['id' => 2, 'name' => 'Technology', 'description' => 'Discuss the latest in tech', 'order' => 2],
                        ['id' => 3, 'name' => 'Gaming', 'description' => 'Video games and gaming culture', 'order' => 3],
                        ['id' => 4, 'name' => 'Support', 'description' => 'Get help with the forum', 'order' => 4],
                        ['id' => 5, 'name' => 'Feedback', 'description' => 'Share your suggestions', 'order' => 5],
                        ['id' => 6, 'name' => 'Off-Topic', 'description' => 'Everything else', 'order' => 6]
                    ]
                ];
                
            case 'settings':
                return [
                    'settings' => [
                        'site_title' => 'Your Forum Name',
                        'site_description' => 'Community Discussion Forum',
                        'default_theme' => 'cyber_nexus',
                        'posts_per_page' => 20,
                        'threads_per_page' => 25,
                        'registration_enabled' => true,
                        'allow_bbcode' => true,
                        'allow_images' => true,
                        'enable_reputation' => true,
                        'enable_signatures' => true,
                        'enable_private_messages' => true,
                        'enable_reports' => true,
                        'max_signature_length' => 100,
                        'post_edit_time_limit' => 900,
                        'maintenance_mode' => false,
                        'admin_email' => 'admin@yourforum.com',
                        'meta_description' => 'Community discussion forum',
                        'meta_keywords' => 'forum,community,discussion',
                        'download_file' => 'forumly_v7.zip',
                        'download_count' => 0,
                        'enable_search' => true
                    ]
                ];
                
            case 'posts':
                return [
                    'last_id' => 1,
                    'posts' => [
                        [
                            'id' => 1,
                            'thread_id' => 1,
                            'user_id' => 1,
                            'content' => 'Welcome to your new forum! This is the first post. You can customize this message in the admin panel.',
                            'created_at' => date('Y-m-d H:i:s'),
                            'edited_at' => null,
                            'edited_by' => null
                        ]
                    ]
                ];
                
            case 'threads':
                return [
                    'last_id' => 1,
                    'threads' => [
                        [
                            'id' => 1,
                            'title' => 'Welcome to Your New Forum!',
                            'user_id' => 1,
                            'category_id' => 1,
                            'created_at' => date('Y-m-d H:i:s'),
                            'updated_at' => date('Y-m-d H:i:s'),
                            'is_sticky' => true,
                            'is_locked' => false,
                            'view_count' => 1
                        ]
                    ]
                ];

            case 'messages':
                return ['last_id' => 0, 'messages' => []];
                
            case 'reports':
                return ['last_id' => 0, 'reports' => []];
                
            case 'downloads':
                return ['total_downloads' => 0, 'downloads' => []];
                
            default:
                return ['last_id' => 0];
        }
    }

    private function loadSettings() {
        $defaultSettings = [
            'site_title' => 'Your Forum Name',
            'site_description' => 'Community Discussion Forum',
            'default_theme' => 'cyber_nexus',
            'posts_per_page' => 20,
            'threads_per_page' => 25,
            'registration_enabled' => true,
            'allow_bbcode' => true,
            'allow_images' => true,
            'enable_reputation' => true,
            'enable_signatures' => true,
            'enable_private_messages' => true,
            'enable_reports' => true,
            'max_signature_length' => 100,
            'post_edit_time_limit' => 900,
            'maintenance_mode' => false,
            'admin_email' => 'admin@yourforum.com',
            'meta_description' => 'Community discussion forum',
            'meta_keywords' => 'forum,community,discussion',
            'download_file' => 'forumly_v7.zip',
            'download_count' => 0,
            'enable_search' => true
        ];

        if (file_exists($this->dataFiles['settings'])) {
            $settingsData = json_decode(file_get_contents($this->dataFiles['settings']), true);
            $this->settings = array_merge($defaultSettings, $settingsData['settings'] ?? []);
        } else {
            $this->settings = $defaultSettings;
        }
    }

    private function authenticateUser() {
        if (isset($_SESSION['user_id'])) {
            $this->currentUser = $this->getUserById($_SESSION['user_id']);
        }
    }

    private function getUserById($id) {
        if (!file_exists($this->dataFiles['users'])) return null;
        $usersData = json_decode(file_get_contents($this->dataFiles['users']), true);
        foreach ($usersData['users'] as $user) {
            if ($user['id'] == $id) return $user;
        }
        return null;
    }

    private function handleRouting() {
        $action = $_GET['action'] ?? 'home';
        
        switch ($action) {
            case 'admin': $this->handleAdmin(); break;
            case 'user': $this->handleUser(); break;
            case 'forum': $this->handleForum(); break;
            case 'messages': $this->handleMessages(); break;
            case 'tos': $this->showTOS(); break;
            default: $this->showHome();
        }
    }

    private function handleAdmin() {
        if (!$this->isAdmin()) {
            $this->showError('Admin access required.');
            return;
        }

        $method = $_GET['method'] ?? 'dashboard';

        switch ($method) {
            case 'dashboard': $this->showAdminDashboard(); break;
            case 'users': $this->handleUserManagement(); break;
            case 'moderators': $this->handleModeratorManagement(); break;
            case 'categories': $this->handleCategoryManagement(); break;
            case 'themes': $this->showThemeManagement(); break;
            case 'settings': $this->handleSettingsManagement(); break;
            case 'reports': $this->showReports(); break;
            case 'system': $this->showSystemInfo(); break;
            default: $this->showAdminDashboard();
        }
    }

    private function handleUser() {
        $method = $_GET['method'] ?? 'login';
        
        if ($_GET['process'] ?? '' == '1') {
            switch ($method) {
                case 'login': $this->processLogin(); break;
                case 'register': $this->processRegistration(); break;
                case 'update-profile': $this->processProfileUpdate(); break;
            }
        } else {
            switch ($method) {
                case 'login': $this->showLoginForm(); break;
                case 'register': $this->showRegisterForm(); break;
                case 'logout': $this->logoutUser(); break;
                case 'profile': $this->showUserProfile(); break;
                default: $this->showLoginForm();
            }
        }
    }

    private function handleForum() {
        $method = $_GET['method'] ?? 'categories';
        
        if ($_GET['process'] ?? '' == '1') {
            switch ($method) {
                case 'new-thread': $this->processNewThread(); break;
                case 'post-reply': $this->processPostReply(); break;
                case 'edit-post': $this->processEditPost(); break;
                case 'report': $this->processReport(); break;
                case 'search': $this->processSearch(); break;
                case 'lock-thread': $this->processLockThread(); break;
                case 'sticky-thread': $this->processStickyThread(); break;
                case 'delete-post': $this->processDeletePost(); break;
            }
        } else {
            switch ($method) {
                case 'categories': $this->showCategories(); break;
                case 'category': $this->showCategory(); break;
                case 'thread': $this->showThread(); break;
                case 'new-thread': $this->showNewThreadForm(); break;
                case 'search': $this->showSearch(); break;
                default: $this->showCategories();
            }
        }
    }

    private function handleMessages() {
        if (!$this->currentUser) {
            header('Location: ?action=user&method=login');
            exit;
        }

        if (!$this->settings['enable_private_messages']) {
            $this->showError('Private messaging is disabled.');
            return;
        }

        $method = $_GET['method'] ?? 'inbox';
        
        if ($_GET['process'] ?? '' == '1') {
            switch ($method) {
                case 'send': $this->processSendMessage(); break;
                case 'delete': $this->processDeleteMessage(); break;
            }
        } else {
            switch ($method) {
                case 'inbox': $this->showInbox(); break;
                case 'sent': $this->showSent(); break;
                case 'compose': $this->showCompose(); break;
                case 'view': $this->showMessage(); break;
                default: $this->showInbox();
            }
        }
    }

    // PERMISSION CHECKS
    private function isAdmin() {
        return $this->currentUser && $this->currentUser['role'] === 'admin';
    }

    private function isModerator() {
        return $this->currentUser && $this->currentUser['role'] === 'moderator';
    }

    private function canModerate() {
        return $this->isAdmin() || $this->isModerator();
    }

    // THEME SYSTEM
    private function getThemeCSS($theme) {
        $themes = [
            'cyber_nexus' => [
                'bg' => 'rgba(16, 16, 32, 0.95)',
                'primary' => '#00ffff',
                'secondary' => '#ff00ff',
                'text' => '#00ffff',
                'accent' => '#ff00ff'
            ],
            'quantum_void' => [
                'bg' => 'rgba(0, 20, 40, 0.95)',
                'primary' => '#00ff88',
                'secondary' => '#0088ff',
                'text' => '#00ff88',
                'accent' => '#0088ff'
            ],
            'neon_dream' => [
                'bg' => 'rgba(40, 0, 50, 0.95)',
                'primary' => '#ff00ff',
                'secondary' => '#ffff00',
                'text' => '#ff00ff',
                'accent' => '#ffff00'
            ],
            'cosmic_horizon' => [
                'bg' => 'rgba(0, 34, 68, 0.95)',
                'primary' => '#ffaa00',
                'secondary' => '#ff5500',
                'text' => '#ffaa00',
                'accent' => '#ff5500'
            ],
            'matrix_reality' => [
                'bg' => 'rgba(0, 34, 0, 0.95)',
                'primary' => '#00ff00',
                'secondary' => '#00ffff',
                'text' => '#00ff00',
                'accent' => '#00ffff'
            ],
            'midnight_ocean' => [
                'bg' => 'rgba(10, 25, 47, 0.95)',
                'primary' => '#4fc3f7',
                'secondary' => '#0277bd',
                'text' => '#81d4fa',
                'accent' => '#0288d1'
            ],
            'sunset_terminal' => [
                'bg' => 'rgba(45, 20, 30, 0.95)',
                'primary' => '#ff6b6b',
                'secondary' => '#feca57',
                'text' => '#ff7979',
                'accent' => '#ee5a6f'
            ]
        ];

        $colors = $themes[$theme] ?? $themes['cyber_nexus'];

        return '
            * { margin: 0; padding: 0; box-sizing: border-box; }
            body { 
                font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, "Helvetica Neue", Arial, sans-serif; 
                line-height: 1.6; 
                min-height: 100vh; 
                background: transparent;
                color: ' . $colors['text'] . ';
            }
            .container { max-width: 1400px; margin: 0 auto; padding: 20px; }
            
            header, nav, .sidebar, footer, .user-panel, .post, .admin-panel, .message-box { 
                background: ' . $colors['bg'] . ';
                padding: 20px; 
                margin-bottom: 20px; 
                border-radius: 12px;
                border: 1px solid ' . $colors['primary'] . ';
                backdrop-filter: blur(10px);
                box-shadow: 0 8px 32px rgba(0,0,0,0.3);
            }
            
            h1, h2, h3 { color: ' . $colors['primary'] . '; text-shadow: 0 0 10px ' . $colors['primary'] . '; }
            a { color: ' . $colors['primary'] . '; text-decoration: none; transition: all 0.3s; }
            a:hover { color: ' . $colors['secondary'] . '; text-shadow: 0 0 5px ' . $colors['secondary'] . '; }
            
            .btn { 
                display: inline-block;
                background: ' . $colors['accent'] . '; 
                color: white; 
                padding: 12px 24px; 
                border: none; 
                border-radius: 6px; 
                text-decoration: none; 
                transition: all 0.3s;
                cursor: pointer;
                font-weight: 600;
                box-shadow: 0 4px 12px rgba(0,0,0,0.2);
            }
            .btn:hover { 
                background: ' . $colors['secondary'] . '; 
                transform: translateY(-2px); 
                box-shadow: 0 6px 20px rgba(0,0,0,0.3);
            }
            
            .btn-small { padding: 6px 12px; font-size: 0.9em; }
            .btn-danger { background: #ff4444; }
            .btn-success { background: #00aa00; }
            
            .main-content { display: flex; gap: 20px; margin-bottom: 20px; }
            .content-area { flex: 1; min-width: 0; }
            .sidebar { width: 320px; flex-shrink: 0; }
            
            .nav-menu { display: flex; list-style: none; gap: 15px; flex-wrap: wrap; }
            .nav-menu li a { 
                display: block;
                padding: 12px 20px; 
                border-radius: 6px; 
                transition: all 0.3s;
                background: rgba(255,255,255,0.05);
            }
            .nav-menu li a:hover { 
                background: rgba(255,255,255,0.1); 
                transform: translateY(-2px);
            }
            
            .form-group { margin-bottom: 20px; }
            .form-group label { display: block; margin-bottom: 8px; font-weight: 600; color: ' . $colors['primary'] . '; }
            .form-group input, .form-group textarea, .form-group select { 
                width: 100%; 
                padding: 12px; 
                background: rgba(0,0,0,0.3); 
                color: ' . $colors['text'] . '; 
                border: 1px solid ' . $colors['primary'] . '; 
                border-radius: 6px;
                font-size: 1em;
                transition: all 0.3s;
            }
            .form-group input:focus, .form-group textarea:focus, .form-group select:focus {
                outline: none;
                border-color: ' . $colors['secondary'] . ';
                box-shadow: 0 0 10px rgba(255,255,255,0.1);
            }
            
            .error { background: #ff4444; color: white; padding: 15px; border-radius: 6px; margin-bottom: 15px; }
            .success { background: #00aa00; color: white; padding: 15px; border-radius: 6px; margin-bottom: 15px; }
            .info { background: #0088ff; color: white; padding: 15px; border-radius: 6px; margin-bottom: 15px; }
            
            .forum-table { width: 100%; border-radius: 10px; overflow: hidden; }
            .forum-table th { 
                background: ' . $colors['accent'] . '; 
                color: white; 
                padding: 12px; 
                text-align: left;
                font-weight: 600;
            }
            .forum-table td { 
                padding: 12px; 
                border-bottom: 1px solid rgba(255,255,255,0.1);
            }
            .forum-table tr:hover { background: rgba(255,255,255,0.05); }
            
            .stats-grid { 
                display: grid; 
                grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); 
                gap: 20px; 
                margin: 20px 0; 
            }
            .stat-card { 
                padding: 30px; 
                border-radius: 12px; 
                text-align: center;
                background: ' . $colors['bg'] . ';
                border: 1px solid ' . $colors['primary'] . ';
                transition: all 0.3s;
            }
            .stat-card:hover {
                transform: translateY(-5px);
                box-shadow: 0 12px 24px rgba(0,0,0,0.3);
            }
            .stat-card h3 { font-size: 2.5em; margin-bottom: 10px; }
            
            .badge { 
                display: inline-block;
                padding: 4px 12px;
                border-radius: 12px;
                font-size: 0.85em;
                font-weight: 600;
                margin-left: 8px;
            }
            .badge-admin { background: #ff00ff; color: white; }
            .badge-mod { background: #00aaff; color: white; }
            
            .signature { 
                margin-top: 15px;
                padding-top: 10px;
                border-top: 1px solid rgba(255,255,255,0.2);
                font-size: 0.9em;
                font-style: italic;
                opacity: 0.8;
            }
            
            .breadcrumb {
                margin-bottom: 20px;
                padding: 10px 15px;
                background: rgba(255,255,255,0.05);
                border-radius: 6px;
            }
            .breadcrumb a { margin: 0 5px; }
            
            .pagination {
                display: flex;
                justify-content: center;
                gap: 10px;
                margin: 20px 0;
            }
            .pagination a, .pagination span {
                padding: 8px 15px;
                border-radius: 6px;
                background: rgba(255,255,255,0.1);
            }
            .pagination .active {
                background: ' . $colors['accent'] . ';
                color: white;
            }
            
            @media (max-width: 968px) { 
                .main-content { flex-direction: column; } 
                .sidebar { width: 100%; }
                .stats-grid { grid-template-columns: 1fr 1fr; }
                .nav-menu { justify-content: center; }
            }
            
            @media (max-width: 480px) {
                .stats-grid { grid-template-columns: 1fr; }
                .container { padding: 10px; }
            }
        ';
    }

    private function getAnimatedBackground() {
        return '
        <style>
            .stars {
                position: fixed;
                top: 0;
                left: 0;
                width: 100%;
                height: 100%;
                z-index: -1;
                background: linear-gradient(135deg, #0a0a12 0%, #1a1a2e 100%);
                overflow: hidden;
            }
            
            .stars:before {
                content: "";
                position: absolute;
                top: 0;
                left: 0;
                width: 100%;
                height: 100%;
                background-image: 
                    radial-gradient(2px 2px at 20px 30px, #eee, transparent),
                    radial-gradient(2px 2px at 40px 70px, #fff, transparent),
                    radial-gradient(1px 1px at 90px 40px, #fff, transparent),
                    radial-gradient(1px 1px at 130px 80px, #fff, transparent),
                    radial-gradient(2px 2px at 160px 30px, #eee, transparent);
                background-repeat: repeat;
                background-size: 200px 100px;
                animation: starsMove 50s linear infinite;
            }
            
            .stars:after {
                content: "";
                position: absolute;
                top: 0;
                left: 0;
                width: 100%;
                height: 100%;
                background-image: 
                    radial-gradient(2px 2px at 120px 120px, #fff, transparent),
                    radial-gradient(2px 2px at 200px 150px, #eee, transparent),
                    radial-gradient(1px 1px at 280px 100px, #fff, transparent);
                background-repeat: repeat;
                background-size: 400px 200px;
                animation: starsMove 80s linear infinite;
            }
            
            @keyframes starsMove {
                0% { transform: translateY(0); }
                100% { transform: translateY(-200px); }
            }
        </style>';
    }

    // RENDERING METHODS
    private function renderPage($title, $content, $sidebar = '') {
        $theme = $_COOKIE['theme'] ?? $this->settings['default_theme'] ?? 'cyber_nexus';
        
        echo '<!DOCTYPE html>
        <html lang="en">
        <head>
            <meta charset="UTF-8">
            <meta name="viewport" content="width=device-width, initial-scale=1.0">
            <meta name="description" content="' . htmlspecialchars($this->settings['meta_description']) . '">
            <meta name="keywords" content="' . htmlspecialchars($this->settings['meta_keywords']) . '">
            <title>' . htmlspecialchars($title) . ' - ' . htmlspecialchars($this->settings['site_title']) . '</title>
            <style>' . $this->getThemeCSS($theme) . '</style>
            ' . $this->getAnimatedBackground() . '
        </head>
        <body>
            <div class="stars"></div>
            <div class="container">
                <header>
                    <h1>' . htmlspecialchars($this->settings['site_title']) . '</h1>
                    <p>' . htmlspecialchars($this->settings['site_description']) . '</p>
                </header>
                
                <nav>' . $this->renderNavigation() . '</nav>
                
                ' . $this->renderUserPanel() . '
                
                <div class="main-content">
                    <div class="content-area">' . $content . '</div>
                    ' . ($sidebar ? '<div class="sidebar">' . $sidebar . '</div>' : '') . '
                </div>
                
                <footer>
                    <p>&copy; 2025 ' . htmlspecialchars($this->settings['site_title']) . ' | Forum Software v' . VERSION . '</p>
                </footer>
            </div>
        </body>
        </html>';
    }

    private function renderNavigation() {
        $menu = '<ul class="nav-menu">';
        $menu .= '<li><a href="?action=home">🏠 Home</a></li>';
        $menu .= '<li><a href="?action=forum&method=categories">💬 Forums</a></li>';
        
        if ($this->currentUser) {
            if ($this->settings['enable_private_messages']) {
                $unreadCount = $this->getUnreadMessageCount();
                $menu .= '<li><a href="?action=messages">✉️ Messages' . ($unreadCount > 0 ? " ($unreadCount)" : '') . '</a></li>';
            }
            if ($this->isAdmin()) {
                $menu .= '<li><a href="?action=admin">⚙️ Admin</a></li>';
            }
            $menu .= '<li><a href="?action=user&method=profile">👤 Profile</a></li>';
            $menu .= '<li><a href="?action=user&method=logout">🚪 Logout</a></li>';
        } else {
            $menu .= '<li><a href="?action=user&method=login">🔐 Login</a></li>';
            if ($this->settings['registration_enabled']) {
                $menu .= '<li><a href="?action=user&method=register">📝 Register</a></li>';
            }
        }
        
        $menu .= '</ul>';
        return $menu;
    }

    private function renderUserPanel() {
        if ($this->currentUser) {
            $badge = '';
            if ($this->isAdmin()) $badge = '<span class="badge badge-admin">ADMIN</span>';
            elseif ($this->isModerator()) $badge = '<span class="badge badge-mod">MODERATOR</span>';
            
            return '<div class="user-panel">Welcome, <strong>' . htmlspecialchars($this->currentUser['username']) . '</strong>' . 
                   $badge . ' | Reputation: ' . $this->currentUser['reputation'] . ' | Posts: ' . $this->currentUser['post_count'] . '</div>';
        } else {
            return '<div class="user-panel">Welcome, Guest! <a href="?action=user&method=login" class="btn btn-small">Login</a>' . 
                   ($this->settings['registration_enabled'] ? ' <a href="?action=user&method=register" class="btn btn-small">Register</a>' : '') . '</div>';
        }
    }

    private function parseBBCode($text) {
        if (!$this->settings['allow_bbcode']) return nl2br(htmlspecialchars($text));
        
        $patterns = [
            '/\[b\](.*?)\[\/b\]/is' => '<strong>$1</strong>',
            '/\[i\](.*?)\[\/i\]/is' => '<em>$1</em>',
            '/\[u\](.*?)\[\/u\]/is' => '<u>$1</u>',
            '/\[url\](.*?)\[\/url\]/is' => '<a href="$1" target="_blank" rel="noopener">$1</a>',
            '/\[url=(.*?)\](.*?)\[\/url\]/is' => '<a href="$1" target="_blank" rel="noopener">$2</a>',
            '/\[img\](.*?)\[\/img\]/is' => $this->settings['allow_images'] ? '<img src="$1" style="max-width:100%;border-radius:8px;">' : '[Image]',
            '/\[code\](.*?)\[\/code\]/is' => '<pre style="background:rgba(0,0,0,0.5);padding:15px;border-radius:6px;overflow-x:auto;"><code>$1</code></pre>',
            '/\[quote\](.*?)\[\/quote\]/is' => '<blockquote style="border-left:3px solid;padding-left:15px;margin:15px 0;opacity:0.8;">$1</blockquote>'
        ];
        
        $text = htmlspecialchars($text);
        $text = preg_replace(array_keys($patterns), array_values($patterns), $text);
        return nl2br($text);
    }

    // HOME PAGE
    private function showHome() {
        $stats = [
            'users' => $this->getUserCount(),
            'threads' => $this->getThreadCount(),
            'posts' => $this->getPostCount(),
            'categories' => $this->getCategoryCount()
        ];

        $latestThreads = $this->getLatestThreads(5);
        $topUsers = $this->getTopUsers(5);

        $threadsList = '';
        foreach ($latestThreads as $thread) {
            $user = $this->getUserById($thread['user_id']);
            $threadsList .= '
                <div class="post" style="margin-bottom:10px;padding:15px;">
                    <strong><a href="?action=forum&method=thread&id=' . $thread['id'] . '">' . htmlspecialchars($thread['title']) . '</a></strong><br>
                    <small>by ' . htmlspecialchars($user['username']) . ' • ' . $this->timeAgo($thread['created_at']) . ' • ' . $thread['view_count'] . ' views</small>
                </div>
            ';
        }

        $usersList = '';
        foreach ($topUsers as $user) {
            $badge = '';
            if ($user['role'] === 'admin') $badge = '<span class="badge badge-admin">ADMIN</span>';
            elseif ($user['role'] === 'moderator') $badge = '<span class="badge badge-mod">MOD</span>';
            
            $usersList .= '
                <div style="padding:10px 0;border-bottom:1px solid rgba(255,255,255,0.1);display:flex;justify-content:space-between;align-items:center;">
                    <div>' . htmlspecialchars($user['username']) . ' ' . $badge . '</div>
                    <div style="text-align:right;">
                        <strong>' . $user['reputation'] . '</strong> rep<br>
                        <small>' . $user['post_count'] . ' posts</small>
                    </div>
                </div>
            ';
        }

        $sidebar = '
            <div class="post">
                <h3>📋 Latest Threads</h3>
                ' . ($threadsList ?: '<p>No threads yet.</p>') . '
            </div>
            <div class="post">
                <h3>🏆 Top Users</h3>
                ' . ($usersList ?: '<p>No users yet.</p>') . '
            </div>
        ';

        $content = '
            <header>
                <h1>Welcome to ' . htmlspecialchars($this->settings['site_title']) . '</h1>
                <p>' . htmlspecialchars($this->settings['site_description']) . '</p>
            </header>
            
            <div class="stats-grid">
                <div class="stat-card">
                    <h3>' . $stats['users'] . '</h3>
                    <p>👥 Members</p>
                </div>
                <div class="stat-card">
                    <h3>' . $stats['threads'] . '</h3>
                    <p>💬 Threads</p>
                </div>
                <div class="stat-card">
                    <h3>' . $stats['posts'] . '</h3>
                    <p>📝 Posts</p>
                </div>
                <div class="stat-card">
                    <h3>' . $stats['categories'] . '</h3>
                    <p>📁 Categories</p>
                </div>
            </div>

            <div style="text-align:center;margin:40px 0;">
                <a href="?action=forum&method=categories" class="btn" style="padding:15px 40px;font-size:1.2em;">Explore Forums</a>
            </div>
        ';
        
        $this->renderPage('Home', $content, $sidebar);
    }

    //    // TOS PAGE
    private function showTOS() {
        $content = '
            <div class="admin-panel">
                <h1>Terms of Service</h1>
                
                <div class="post">
                    <h2>1. Acceptance of Terms</h2>
                    <p>By accessing and using this forum, you accept and agree to be bound by the terms and provision of this agreement.</p>
                </div>

                <div class="post">
                    <h2>2. User Responsibilities</h2>
                    <p>Users are responsible for their own content and behavior. Prohibited activities include:</p>
                    <ul>
                        <li>Spamming or flooding</li>
                        <li>Harassing other users</li>
                        <li>Posting illegal content</li>
                        <li>Impersonating others</li>
                    </ul>
                </div>

                <div class="post">
                    <h2>3. Content Ownership</h2>
                    <p>Users retain ownership of their content but grant the forum a license to display and distribute it within the forum platform.</p>
                </div>

                <div class="post">
                    <h2>4. Privacy</h2>
                    <p>We respect your privacy. User data is stored locally and not shared with third parties without consent.</p>
                </div>

                <div class="post">
                    <h2>5. Moderation</h2>
                    <p>Administrators and moderators reserve the right to moderate content, lock or delete threads, ban users, and take any action necessary to maintain forum quality.</p>
                </div>

                <div style="text-align:center;margin-top:20px;">
                    <a href="?action=home" class="btn">Back to Home</a>
                </div>
            </div>
        ';
        $this->renderPage('Terms of Service', $content);
    }

    // CATEGORIES
    private function showCategories() {
        $categoriesData = json_decode(file_get_contents($this->dataFiles['categories']), true);
        $categories = $categoriesData['categories'];
        
        usort($categories, function($a, $b) {
            return $a['order'] - $b['order'];
        });
        
        $categoryList = '';
        foreach ($categories as $category) {
            $threadCount = $this->getThreadCountByCategory($category['id']);
            $latestThread = $this->getLatestThreadByCategory($category['id']);
            $latestInfo = '';
            
            if ($latestThread) {
                $user = $this->getUserById($latestThread['user_id']);
                $latestInfo = '
                    <div style="font-size:0.9em;margin-top:10px;">
                        Latest: <a href="?action=forum&method=thread&id=' . $latestThread['id'] . '">' . htmlspecialchars($latestThread['title']) . '</a><br>
                        by ' . htmlspecialchars($user['username']) . ' • ' . $this->timeAgo($latestThread['created_at']) . '
                    </div>
                ';
            }
            
            $categoryList .= '
                <div class="post">
                    <div style="display:flex;justify-content:space-between;align-items:flex-start;">
                        <div style="flex:1;">
                            <h3>📁 ' . htmlspecialchars($category['name']) . '</h3>
                            <p>' . htmlspecialchars($category['description']) . '</p>
                            ' . $latestInfo . '
                        </div>
                        <div style="text-align:right;min-width:100px;">
                            <div style="font-size:1.5em;font-weight:bold;">' . $threadCount . '</div>
                            <div style="font-size:0.9em;">threads</div>
                        </div>
                    </div>
                    <div style="margin-top:15px;">
                        <a href="?action=forum&method=category&id=' . $category['id'] . '" class="btn">Browse</a>
                        ' . ($this->currentUser ? '<a href="?action=forum&method=new-thread&category=' . $category['id'] . '" class="btn">New Thread</a>' : '') . '
                    </div>
                </div>
            ';
        }
        
        $sidebar = '
            <div class="post">
                <h3>📊 Forum Statistics</h3>
                <p>Threads: ' . $this->getThreadCount() . '</p>
                <p>Posts: ' . $this->getPostCount() . '</p>
                <p>Users: ' . $this->getUserCount() . '</p>
                <p>Categories: ' . $this->getCategoryCount() . '</p>
            </div>
            ' . ($this->settings['enable_search'] ? '
            <div class="post">
                <h3>🔍 Search Forums</h3>
                <form method="GET" action="?action=forum&method=search&process=1">
                    <input type="hidden" name="action" value="forum">
                    <input type="hidden" name="method" value="search">
                    <input type="hidden" name="process" value="1">
                    <div class="form-group">
                        <input type="text" name="q" placeholder="Search threads..." required>
                    </div>
                    <button type="submit" class="btn" style="width:100%;">Search</button>
                </form>
            </div>' : '') . '
            ' . ($this->currentUser ? '
            <div class="post">
                <h3>⚡ Quick Actions</h3>
                <a href="?action=forum&method=new-thread" class="btn" style="display:block;text-align:center;margin:5px 0;">Start New Thread</a>
                ' . ($this->settings['enable_private_messages'] ? '<a href="?action=messages" class="btn" style="display:block;text-align:center;margin:5px 0;">Messages</a>' : '') . '
            </div>' : '') . '
        ';
        
        $content = '
            <h2>💬 Forum Categories</h2>
            ' . $categoryList . '
        ';
        
        $this->renderPage('Categories', $content, $sidebar);
    }

    // CATEGORY VIEW
    private function showCategory() {
        $categoryId = (int)($_GET['id'] ?? 0);
        $category = $this->getCategoryById($categoryId);
        
        if (!$category) {
            $this->showError('Category not found.');
            return;
        }
        
        $threads = $this->getThreadsByCategory($categoryId);
        
        // Sort by sticky first, then by updated_at
        usort($threads, function($a, $b) {
            if ($a['is_sticky'] != $b['is_sticky']) {
                return $b['is_sticky'] - $a['is_sticky'];
            }
            return strtotime($b['updated_at']) - strtotime($a['updated_at']);
        });
        
        $threadList = '';
        foreach ($threads as $thread) {
            $user = $this->getUserById($thread['user_id']);
            $replyCount = $this->getReplyCount($thread['id']);
            $latestPost = $this->getLatestPostByThread($thread['id']);
            $latestInfo = '';
            
            if ($latestPost) {
                $latestUser = $this->getUserById($latestPost['user_id']);
                $latestInfo = '
                    <div style="font-size:0.9em;">
                        Latest by ' . htmlspecialchars($latestUser['username']) . '<br>
                        ' . $this->timeAgo($latestPost['created_at']) . '
                    </div>
                ';
            }
            
            $stickyIcon = $thread['is_sticky'] ? '📌 ' : '';
            $lockedIcon = $thread['is_locked'] ? '🔒 ' : '';
            
            $threadList .= '
                <div class="post">
                    <div style="display:flex;justify-content:space-between;align-items:center;">
                        <div style="flex:1;">
                            <h4>' . $stickyIcon . $lockedIcon . '<a href="?action=forum&method=thread&id=' . $thread['id'] . '">' . htmlspecialchars($thread['title']) . '</a></h4>
                            <p>by ' . htmlspecialchars($user['username']) . ' • ' . $this->timeAgo($thread['created_at']) . ' • ' . $replyCount . ' replies • ' . $thread['view_count'] . ' views</p>
                        </div>
                        <div style="text-align:right;min-width:150px;">
                            ' . $latestInfo . '
                        </div>
                    </div>
                </div>
            ';
        }
        
        $breadcrumb = '
            <div class="breadcrumb">
                <a href="?action=home">Home</a> → 
                <a href="?action=forum&method=categories">Forums</a> → 
                ' . htmlspecialchars($category['name']) . '
            </div>
        ';
        
        $content = '
            ' . $breadcrumb . '
            <div style="display:flex;justify-content:space-between;align-items:center;margin-bottom:20px;">
                <div>
                    <h2>📁 ' . htmlspecialchars($category['name']) . '</h2>
                    <p>' . htmlspecialchars($category['description']) . '</p>
                </div>
                ' . ($this->currentUser ? '<a href="?action=forum&method=new-thread&category=' . $categoryId . '" class="btn">New Thread</a>' : '') . '
            </div>
            ' . ($threadList ?: '<div class="info">No threads in this category yet. Be the first to post!</div>') . '
        ';
        
        $this->renderPage($category['name'], $content);
    }

    // THREAD VIEW
    private function showThread() {
        $threadId = (int)($_GET['id'] ?? 0);
        $thread = $this->getThreadById($threadId);
        
        if (!$thread) {
            $this->showError('Thread not found.');
            return;
        }
        
        $this->incrementViewCount($threadId);
        $category = $this->getCategoryById($thread['category_id']);
        $author = $this->getUserById($thread['user_id']);
        $posts = $this->getPostsByThread($threadId);
        
        $postsList = '';
        foreach ($posts as $post) {
            $postUser = $this->getUserById($post['user_id']);
            $badge = '';
            if ($postUser['role'] === 'admin') $badge = '<span class="badge badge-admin">ADMIN</span>';
            elseif ($postUser['role'] === 'moderator') $badge = '<span class="badge badge-mod">MOD</span>';
            
            $editedInfo = '';
            if ($post['edited_at']) {
                $editedBy = $this->getUserById($post['edited_by']);
                $editedInfo = '<div style="font-size:0.85em;opacity:0.7;margin-top:10px;"><em>Last edited by ' . htmlspecialchars($editedBy['username']) . ' on ' . date('M j, Y g:i A', strtotime($post['edited_at'])) . '</em></div>';
            }
            
            $canEdit = $this->currentUser && ($this->currentUser['id'] == $postUser['id'] || $this->canModerate());
            $canDelete = $this->canModerate() && $post['id'] != $posts[0]['id']; // Can't delete first post
            
            $canEditTime = true;
            if ($this->currentUser && $this->currentUser['id'] == $postUser['id'] && !$this->canModerate()) {
                $postAge = time() - strtotime($post['created_at']);
                $canEditTime = $postAge < $this->settings['post_edit_time_limit'];
            }
            
            $postsList .= '
                <div class="post" id="post-' . $post['id'] . '">
                    <div style="display:flex;justify-content:space-between;border-bottom:1px solid rgba(255,255,255,0.2);padding-bottom:10px;margin-bottom:10px;">
                        <div>
                            <strong>' . htmlspecialchars($postUser['username']) . '</strong> ' . $badge . '
                            <div style="font-size:0.85em;opacity:0.8;">Posts: ' . $postUser['post_count'] . ' | Rep: ' . $postUser['reputation'] . '</div>
                        </div>
                        <div style="text-align:right;">
                            <div>' . $this->timeAgo($post['created_at']) . '</div>
                            <a href="#post-' . $post['id'] . '" style="font-size:0.9em;">#' . $post['id'] . '</a>
                        </div>
                    </div>
                    <div style="margin-top:10px;">' . $this->parseBBCode($post['content']) . '</div>
                    ' . ($postUser['signature'] && $this->settings['enable_signatures'] ? '<div class="signature">' . htmlspecialchars($postUser['signature']) . '</div>' : '') . '
                    ' . $editedInfo . '
                    <div style="margin-top:10px;">
                        ' . ($canEdit && $canEditTime ? '<a href="?action=forum&method=edit-post&id=' . $post['id'] . '" class="btn btn-small">Edit</a> ' : '') . '
                        ' . ($canDelete ? '<a href="?action=forum&method=delete-post&process=1&id=' . $post['id'] . '&thread=' . $threadId . '" class="btn btn-small btn-danger" onclick="return confirm(\'Delete this post?\')">Delete</a> ' : '') . '
                        ' . ($this->currentUser && $this->settings['enable_reports'] ? '<a href="?action=forum&method=report&post=' . $post['id'] . '" class="btn btn-small">Report</a>' : '') . '
                    </div>
                </div>
            ';
        }
        
        $modTools = '';
        if ($this->canModerate()) {
            $modTools = '
                <div class="mod-tools">
                    <h4>🛡️ Moderator Tools</h4>
                    <a href="?action=forum&method=lock-thread&process=1&id=' . $threadId . '" class="btn btn-small">' . ($thread['is_locked'] ? 'Unlock' : 'Lock') . ' Thread</a>
                    <a href="?action=forum&method=sticky-thread&process=1&id=' . $threadId . '" class="btn btn-small">' . ($thread['is_sticky'] ? 'Unpin' : 'Pin') . ' Thread</a>
                </div>
            ';
        }
        
        $replyForm = '';
        if ($this->currentUser) {
            if (!$thread['is_locked'] || $this->canModerate()) {
                $replyForm = '
                    <div class="post">
                        <h3>💬 Post Reply</h3>
                        <form method="POST" action="?action=forum&method=post-reply&process=1&thread=' . $threadId . '">
                            <div class="form-group">
                                <textarea name="content" rows="6" placeholder="Your reply..." required></textarea>
                            </div>
                            <button type="submit" class="btn">Post Reply</button>
                        </form>
                    </div>
                ';
            } else {
                $replyForm = '<div class="error">🔒 This thread is locked. Only moderators can post.</div>';
            }
        } else {
            $replyForm = '<div class="info">Please <a href="?action=user&method=login">login</a> to post replies.</div>';
        }
        
        $breadcrumb = '
            <div class="breadcrumb">
                <a href="?action=home">Home</a> → 
                <a href="?action=forum&method=categories">Forums</a> → 
                <a href="?action=forum&method=category&id=' . $category['id'] . '">' . htmlspecialchars($category['name']) . '</a> → 
                ' . htmlspecialchars($thread['title']) . '
            </div>
        ';
        
        $content = '
            ' . $breadcrumb . '
            <div class="post">
                <h2>' . ($thread['is_sticky'] ? '📌 ' : '') . ($thread['is_locked'] ? '🔒 ' : '') . htmlspecialchars($thread['title']) . '</h2>
                <p>by ' . htmlspecialchars($author['username']) . ' in <a href="?action=forum&method=category&id=' . $category['id'] . '">' . htmlspecialchars($category['name']) . '</a> • ' . $this->timeAgo($thread['created_at']) . ' • ' . count($posts) . ' posts • ' . $thread['view_count'] . ' views</p>
            </div>
            ' . $modTools . '
            ' . $postsList . '
            ' . $replyForm . '
        ';
        
        $this->renderPage($thread['title'], $content);
    }

    // NEW THREAD FORM
    private function showNewThreadForm() {
        if (!$this->currentUser) {
            header('Location: ?action=user&method=login');
            exit;
        }
        
        $categoryId = (int)($_GET['category'] ?? 0);
        $categoriesData = json_decode(file_get_contents($this->dataFiles['categories']), true);
        $categories = $categoriesData['categories'];
        
        $categoryOptions = '';
        foreach ($categories as $category) {
            $selected = $category['id'] == $categoryId ? 'selected' : '';
            $categoryOptions .= '<option value="' . $category['id'] . '" ' . $selected . '>' . htmlspecialchars($category['name']) . '</option>';
        }
        
        $content = '
            <div class="admin-panel">
                <h2>✍️ Create New Thread</h2>
                <form method="POST" action="?action=forum&method=new-thread&process=1">
                    <div class="form-group">
                        <label>Thread Title</label>
                        <input type="text" name="title" required maxlength="200" placeholder="Enter thread title...">
                    </div>
                    <div class="form-group">
                        <label>Category</label>
                        <select name="category_id" required>
                            <option value="">Select Category</option>
                            ' . $categoryOptions . '
                        </select>
                    </div>
                    <div class="form-group">
                        <label>Content</label>
                        <textarea name="content" rows="12" required placeholder="Write your post content..."></textarea>
                    </div>
                    ' . ($this->settings['allow_bbcode'] ? '
                    <div class="info" style="font-size:0.9em;">
                        <strong>BBCode:</strong> [b]bold[/b] [i]italic[/i] [u]underline[/u] [url]link[/url] [img]image[/img] [code]code[/code] [quote]quote[/quote]
                    </div>' : '') . '
                    <button type="submit" class="btn">Create Thread</button>
                    <a href="?action=forum&method=categories" class="btn" style="background:#666;">Cancel</a>
                </form>
            </div>
        ';
        
        $this->renderPage('New Thread', $content);
    }

    // PROCESS NEW THREAD
    private function processNewThread() {
        if (!$this->currentUser) {
            $this->showError('You must be logged in to create threads.');
            return;
        }
        
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $title = trim($_POST['title'] ?? '');
            $content = trim($_POST['content'] ?? '');
            $categoryId = (int)($_POST['category_id'] ?? 0);
            
            if (empty($title) || empty($content) || $categoryId <= 0) {
                $this->showError('Please fill in all fields.');
                return;
            }
            
            if (strlen($title) > 200) {
                $this->showError('Title must be less than 200 characters.');
                return;
            }
            
            // Create thread
            $threadsData = json_decode(file_get_contents($this->dataFiles['threads']), true);
            $newThreadId = ++$threadsData['last_id'];
            
            $newThread = [
                'id' => $newThreadId,
                'title' => $title,
                'user_id' => $this->currentUser['id'],
                'category_id' => $categoryId,
                'created_at' => date('Y-m-d H:i:s'),
                'updated_at' => date('Y-m-d H:i:s'),
                'is_sticky' => false,
                'is_locked' => false,
                'view_count' => 0
            ];
            
            $threadsData['threads'][] = $newThread;
            file_put_contents($this->dataFiles['threads'], json_encode($threadsData, JSON_PRETTY_PRINT));
            
            // Create first post
            $postsData = json_decode(file_get_contents($this->dataFiles['posts']), true);
            $newPostId = ++$postsData['last_id'];
            
            $newPost = [
                'id' => $newPostId,
                'thread_id' => $newThreadId,
                'user_id' => $this->currentUser['id'],
                'content' => $content,
                'created_at' => date('Y-m-d H:i:s'),
                'edited_at' => null,
                'edited_by' => null
            ];
            
            $postsData['posts'][] = $newPost;
            file_put_contents($this->dataFiles['posts'], json_encode($postsData, JSON_PRETTY_PRINT));
            
            // Update user post count
            $this->incrementUserPostCount($this->currentUser['id']);
            
            header('Location: ?action=forum&method=thread&id=' . $newThreadId);
            exit;
        }
    }

    // PROCESS REPLY
    private function processPostReply() {
        if (!$this->currentUser) {
            $this->showError('You must be logged in to post replies.');
            return;
        }
        
        $threadId = (int)($_GET['thread'] ?? 0);
        $thread = $this->getThreadById($threadId);
        
        if (!$thread) {
            $this->showError('Thread not found.');
            return;
        }
        
        if ($thread['is_locked'] && !$this->canModerate()) {
            $this->showError('This thread is locked. You cannot post replies.');
            return;
        }
        
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $content = trim($_POST['content'] ?? '');
            
            if (empty($content)) {
                $this->showError('Content is required.');
                return;
            }
            
            $postsData = json_decode(file_get_contents($this->dataFiles['posts']), true);
            $newPostId = ++$postsData['last_id'];
            
            $newPost = [
                'id' => $newPostId,
                'thread_id' => $threadId,
                'user_id' => $this->currentUser['id'],
                'content' => $content,
                'created_at' => date('Y-m-d H:i:s'),
                'edited_at' => null,
                'edited_by' => null
            ];
            
            $postsData['posts'][] = $newPost;
            file_put_contents($this->dataFiles['posts'], json_encode($postsData, JSON_PRETTY_PRINT));
            
            // Update thread's updated_at
            $threadsData = json_decode(file_get_contents($this->dataFiles['threads']), true);
            foreach ($threadsData['threads'] as &$t) {
                if ($t['id'] == $threadId) {
                    $t['updated_at'] = date('Y-m-d H:i:s');
                    break;
                }
            }
            file_put_contents($this->dataFiles['threads'], json_encode($threadsData, JSON_PRETTY_PRINT));
            
            // Update user post count
            $this->incrementUserPostCount($this->currentUser['id']);
            
            header('Location: ?action=forum&method=thread&id=' . $threadId . '#post-' . $newPostId);
            exit;
        }
    }

    // PROCESS EDIT POST
    private function processEditPost() {
        if (!$this->currentUser) {
            $this->showError('You must be logged in.');
            return;
        }
        
        $postId = (int)($_GET['id'] ?? 0);
        $post = $this->getPostById($postId);
        
        if (!$post) {
            $this->showError('Post not found.');
            return;
        }
        
        $canEdit = $this->currentUser['id'] == $post['user_id'] || $this->canModerate();
        
        if (!$canEdit) {
            $this->showError('You cannot edit this post.');
            return;
        }
        
        // Check time limit for non-mods
        if ($this->currentUser['id'] == $post['user_id'] && !$this->canModerate()) {
            $postAge = time() - strtotime($post['created_at']);
            if ($postAge > $this->settings['post_edit_time_limit']) {
                $this->showError('Edit time limit expired (' . ($this->settings['post_edit_time_limit'] / 60) . ' minutes).');
                return;
            }
        }
        
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $content = trim($_POST['content'] ?? '');
            
            if (empty($content)) {
                $this->showError('Content is required.');
                return;
            }
            
            $postsData = json_decode(file_get_contents($this->dataFiles['posts']), true);
            foreach ($postsData['posts'] as &$p) {
                if ($p['id'] == $postId) {
                    $p['content'] = $content;
                    $p['edited_at'] = date('Y-m-d H:i:s');
                    $p['edited_by'] = $this->currentUser['id'];
                    break;
                }
            }
            file_put_contents($this->dataFiles['posts'], json_encode($postsData, JSON_PRETTY_PRINT));
            
            header('Location: ?action=forum&method=thread&id=' . $post['thread_id'] . '#post-' . $postId);
            exit;
        } else {
            // Show edit form
            $content = '
                <div class="admin-panel">
                    <h2>✏️ Edit Post</h2>
                    <form method="POST">
                        <div class="form-group">
                            <label>Content</label>
                            <textarea name="content" rows="10" required>' . htmlspecialchars($post['content']) . '</textarea>
                        </div>
                        <button type="submit" class="btn">Save Changes</button>
                        <a href="?action=forum&method=thread&id=' . $post['thread_id'] . '" class="btn" style="background:#666;">Cancel</a>
                    </form>
                </div>
            ';
            $this->renderPage('Edit Post', $content);
        }
    }

    // PROCESS DELETE POST
    private function processDeletePost() {
        if (!$this->canModerate()) {
            $this->showError('Moderator access required.');
            return;
        }
        
        $postId = (int)($_GET['id'] ?? 0);
        $threadId = (int)($_GET['thread'] ?? 0);
        
        $postsData = json_decode(file_get_contents($this->dataFiles['posts']), true);
        $postsData['posts'] = array_filter($postsData['posts'], function($p) use ($postId) {
            return $p['id'] != $postId;
        });
        $postsData['posts'] = array_values($postsData['posts']);
        file_put_contents($this->dataFiles['posts'], json_encode($postsData, JSON_PRETTY_PRINT));
        
        header('Location: ?action=forum&method=thread&id=' . $threadId);
        exit;
    }

    // PROCESS LOCK THREAD
    private function processLockThread() {
        if (!$this->canModerate()) {
            $this->showError('Moderator access required.');
            return;
        }
        
        $threadId = (int)($_GET['id'] ?? 0);
        
        $threadsData = json_decode(file_get_contents($this->dataFiles['threads']), true);
        foreach ($threadsData['threads'] as &$thread) {
            if ($thread['id'] == $threadId) {
                $thread['is_locked'] = !$thread['is_locked'];
                break;
            }
        }
        file_put_contents($this->dataFiles['threads'], json_encode($threadsData, JSON_PRETTY_PRINT));
        
        header('Location: ?action=forum&method=thread&id=' . $threadId);
        exit;
    }

    // PROCESS STICKY THREAD
    private function processStickyThread() {
        if (!$this->canModerate()) {
            $this->showError('Moderator access required.');
            return;
        }
        
        $threadId = (int)($_GET['id'] ?? 0);
        
        $threadsData = json_decode(file_get_contents($this->dataFiles['threads']), true);
        foreach ($threadsData['threads'] as &$thread) {
            if ($thread['id'] == $threadId) {
                $thread['is_sticky'] = !$thread['is_sticky'];
                break;
            }
        }
        file_put_contents($this->dataFiles['threads'], json_encode($threadsData, JSON_PRETTY_PRINT));
        
        header('Location: ?action=forum&method=thread&id=' . $threadId);
        exit;
    }

    // SEARCH
    private function showSearch() {
        $content = '
            <div class="admin-panel">
                <h2>🔍 Search Forums</h2>
                <form method="GET" action="?action=forum&method=search&process=1">
                    <input type="hidden" name="action" value="forum">
                    <input type="hidden" name="method" value="search">
                    <input type="hidden" name="process" value="1">
                    <div class="form-group">
                        <label>Search Query</label>
                        <input type="text" name="q" placeholder="Enter search terms..." required>
                    </div>
                    <button type="submit" class="btn">Search</button>
                </form>
            </div>
        ';
        $this->renderPage('Search', $content);
    }

    private function processSearch() {
        $query = trim($_GET['q'] ?? '');
        
        if (empty($query)) {
            $this->showError('Please enter a search query.');
            return;
        }
        
        $threadsData = json_decode(file_get_contents($this->dataFiles['threads']), true);
        $results = [];
        
        foreach ($threadsData['threads'] as $thread) {
            if (stripos($thread['title'], $query) !== false) {
                $results[] = $thread;
            }
        }
        
        $resultsList = '';
        if (empty($results)) {
            $resultsList = '<div class="info">No threads found matching "' . htmlspecialchars($query) . '"</div>';
        } else {
            foreach ($results as $thread) {
                $user = $this->getUserById($thread['user_id']);
                $category = $this->getCategoryById($thread['category_id']);
                $resultsList .= '
                    <div class="post">
                        <h4><a href="?action=forum&method=thread&id=' . $thread['id'] . '">' . htmlspecialchars($thread['title']) . '</a></h4>
                        <p>in <a href="?action=forum&method=category&id=' . $category['id'] . '">' . htmlspecialchars($category['name']) . '</a> • by ' . htmlspecialchars($user['username']) . ' • ' . $this->timeAgo($thread['created_at']) . '</p>
                    </div>
                ';
            }
        }
        
        $content = '
            <div class="admin-panel">
                <h2>🔍 Search Results for "' . htmlspecialchars($query) . '"</h2>
                <p>Found ' . count($results) . ' thread(s)</p>
                <form method="GET" action="?action=forum&method=search&process=1" style="margin:20px 0;">
                    <input type="hidden" name="action" value="forum">
                    <input type="hidden" name="method" value="search">
                    <input type="hidden" name="process" value="1">
                    <div class="form-group">
                        <input type="text" name="q" value="' . htmlspecialchars($query) . '" placeholder="Search again...">
                    </div>
                    <button type="submit" class="btn">Search</button>
                </form>
            </div>
            ' . $resultsList . '
        ';
        
        $this->renderPage('Search Results', $content);
    }

    // PROCESS REPORT
    private function processReport() {
        if (!$this->currentUser || !$this->settings['enable_reports']) {
            $this->showError('You cannot report posts.');
            return;
        }
        
        $postId = (int)($_GET['post'] ?? 0);
        $post = $this->getPostById($postId);
        
        if (!$post) {
            $this->showError('Post not found.');
            return;
        }
        
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $reason = trim($_POST['reason'] ?? '');
            
            if (empty($reason)) {
                $this->showError('Please provide a reason.');
                return;
            }
            
            $reportsData = json_decode(file_get_contents($this->dataFiles['reports']), true);
            $newReportId = ++$reportsData['last_id'];
            
            $newReport = [
                'id' => $newReportId,
                'post_id' => $postId,
                'thread_id' => $post['thread_id'],
                'reported_by' => $this->currentUser['id'],
                'reason' => $reason,
                'created_at' => date('Y-m-d H:i:s'),
                'status' => 'pending'
            ];
            
            $reportsData['reports'][] = $newReport;
            file_put_contents($this->dataFiles['reports'], json_encode($reportsData, JSON_PRETTY_PRINT));
            
            $this->showSuccess('Report submitted successfully. Moderators will review it.');
        } else {
            // Show report form
            $content = '
                <div class="admin-panel">
                    <h2>🚨 Report Post</h2>
                    <p>You are reporting a post. Please provide a detailed reason.</p>
                    <form method="POST">
                        <div class="form-group">
                            <label>Reason for Report</label>
                            <textarea name="reason" rows="5" required placeholder="Explain why this post should be reviewed..."></textarea>
                        </div>
                        <button type="submit" class="btn">Submit Report</button>
                        <a href="?action=forum&method=thread&id=' . $post['thread_id'] . '" class="btn" style="background:#666;">Cancel</a>
                    </form>
                </div>
            ';
            $this->renderPage('Report Post', $content);
        }
    }

    // USER REGISTRATION
    private function showLoginForm() {
        $content = '
            <div class="admin-panel" style="max-width:500px;margin:0 auto;">
                <h2>🔐 Login</h2>
                <form method="POST" action="?action=user&method=login&process=1">
                    <div class="form-group">
                        <label>Username</label>
                        <input type="text" name="username" required autocomplete="username">
                    </div>
                    <div class="form-group">
                        <label>Password</label>
                        <input type="password" name="password" required autocomplete="current-password">
                    </div>
                    <button type="submit" class="btn" style="width:100%;">Login</button>
                </form>
                <p style="margin-top:20px;text-align:center;">
                    ' . ($this->settings['registration_enabled'] ? 'Don\'t have an account? <a href="?action=user&method=register">Register here</a><br>' : '') . '
                    <a href="?action=home">Back to Home</a>
                </p>
            </div>
        ';
        $this->renderPage('Login', $content);
    }

    private function showRegisterForm() {
        if (!$this->settings['registration_enabled']) {
            $this->showError('Registration is currently disabled.');
            return;
        }
        
        $content = '
            <div class="admin-panel" style="max-width:500px;margin:0 auto;">
                <h2>📝 Register</h2>
                <form method="POST" action="?action=user&method=register&process=1">
                    <div class="form-group">
                        <label>Username</label>
                        <input type="text" name="username" required minlength="3" maxlength="20" autocomplete="username">
                    </div>
                    <div class="form-group">
                        <label>Email</label>
                        <input type="email" name="email" required autocomplete="email">
                    </div>
                    <div class="form-group">
                        <label>Password</label>
                        <input type="password" name="password" required minlength="6" autocomplete="new-password">
                    </div>
                    <div class="form-group">
                        <label>Confirm Password</label>
                        <input type="password" name="confirm_password" required autocomplete="new-password">
                    </div>
                    <button type="submit" class="btn" style="width:100%;">Create Account</button>
                </form>
                <p style="margin-top:20px;text-align:center;">
                    Already have an account? <a href="?action=user&method=login">Login here</a>
                </p>
            </div>
        ';
        $this->renderPage('Register', $content);
    }

    private function processLogin() {
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $username = trim($_POST['username'] ?? '');
            $password = $_POST['password'] ?? '';
            
            $usersData = json_decode(file_get_contents($this->dataFiles['users']), true);
            
            foreach ($usersData['users'] as &$user) {
                if ($user['username'] === $username && password_verify($password, $user['password_hash'])) {
                    if ($user['is_banned']) {
                        $this->showError('Your account has been banned.');
                        return;
                    }
                    
                    $_SESSION['user_id'] = $user['id'];
                    $this->currentUser = $user;
                    
                    // Update last login
                    $user['last_login'] = date('Y-m-d H:i:s');
                    file_put_contents($this->dataFiles['users'], json_encode($usersData, JSON_PRETTY_PRINT));
                    
                    header('Location: ?action=home');
                    exit;
                }
            }
            
            $this->showError('Invalid username or password.');
        }
    }

    private function processRegistration() {
        if (!$this->settings['registration_enabled']) {
            $this->showError('Registration is currently disabled.');
            return;
        }
        
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $username = trim($_POST['username'] ?? '');
            $email = trim($_POST['email'] ?? '');
            $password = $_POST['password'] ?? '';
            $confirm = $_POST['confirm_password'] ?? '';
            
            if ($password !== $confirm) {
                $this->showError('Passwords do not match.');
                return;
            }
            
            if (strlen($username) < 3 || strlen($username) > 20) {
                $this->showError('Username must be 3-20 characters.');
                return;
            }
            
            if (strlen($password) < 6) {
                $this->showError('Password must be at least 6 characters.');
                return;
            }
            
            if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
                $this->showError('Invalid email address.');
                return;
            }
            
            $usersData = json_decode(file_get_contents($this->dataFiles['users']), true);
            
            foreach ($usersData['users'] as $user) {
                if (strtolower($user['username']) === strtolower($username)) {
                    $this->showError('Username is already taken.');
                    return;
                }
                if (strtolower($user['email']) === strtolower($email)) {
                    $this->showError('Email is already registered.');
                    return;
                }
            }
            
            $newUser = [
                'id' => ++$usersData['last_id'],
                'username' => $username,
                'email' => $email,
                'password_hash' => password_hash($password, PASSWORD_DEFAULT),
                'role' => 'user',
                'joined_date' => date('Y-m-d H:i:s'),
                'reputation' => 0,
                'signature' => '',
                'avatar' => '',
                'is_banned' => false,
                'last_login' => date('Y-m-d H:i:s'),
                'post_count' => 0
            ];
            
            $usersData['users'][] = $newUser;
            file_put_contents($this->dataFiles['users'], json_encode($usersData, JSON_PRETTY_PRINT));
            
            $_SESSION['user_id'] = $newUser['id'];
            $this->currentUser = $newUser;
            
            header('Location: ?action=home');
            exit;
        }
    }

    private function logoutUser() {
        session_destroy();
        header('Location: ?action=home');
        exit;
    }

    // USER PROFILE
    private function showUserProfile() {
        if (!$this->currentUser) {
            header('Location: ?action=user&method=login');
            exit;
        }
        
        if (isset($_POST['update_signature'])) {
            $this->processProfileUpdate();
            return;
        }
        
        $userThreads = $this->getUserThreads($this->currentUser['id'], 10);
        
        $threadsList = '';
        foreach ($userThreads as $thread) {
            $category = $this->getCategoryById($thread['category_id']);
            $threadsList .= '<div style="padding:8px 0;border-bottom:1px solid rgba(255,255,255,0.1);"><a href="?action=forum&method=thread&id=' . $thread['id'] . '">' . htmlspecialchars($thread['title']) . '</a><br><small>' . htmlspecialchars($category['name']) . ' • ' . $this->timeAgo($thread['created_at']) . '</small></div>';
        }
        
        $content = '
            <div class="admin-panel">
                <h2>👤 User Profile</h2>
                <div style="display:grid;grid-template-columns:1fr 1fr;gap:20px;">
                    <div class="post">
                        <h3>Account Information</h3>
                        <p><strong>Username:</strong> ' . htmlspecialchars($this->currentUser['username']) . '</p>
                        <p><strong>Email:</strong> ' . htmlspecialchars($this->currentUser['email']) . '</p>
                        <p><strong>Role:</strong> ' . ucfirst($this->currentUser['role']) . '</p>
                        <p><strong>Reputation:</strong> ' . $this->currentUser['reputation'] . '</p>
                        <p><strong>Post Count:</strong> ' . $this->currentUser['post_count'] . '</p>
                        <p><strong>Joined:</strong> ' . date('M j, Y', strtotime($this->currentUser['joined_date'])) . '</p>
                        <p><strong>Last Login:</strong> ' . ($this->currentUser['last_login'] ? date('M j, Y g:i A', strtotime($this->currentUser['last_login'])) : 'Never') . '</p>
                    </div>
                    
                    ' . ($this->settings['enable_signatures'] ? '
                    <div class="post">
                        <h3>Signature</h3>
                        <form method="POST">
                            <div class="form-group">
                                <textarea name="signature" maxlength="' . $this->settings['max_signature_length'] . '" placeholder="Your signature (max ' . $this->settings['max_signature_length'] . ' chars)">' . htmlspecialchars($this->currentUser['signature']) . '</textarea>
                            </div>
                            <button type="submit" name="update_signature" class="btn">Update Signature</button>
                        </form>
                    </div>' : '') . '
                </div>
                
                <div class="post" style="margin-top:20px;">
                    <h3>📋 Your Recent Threads</h3>
                    ' . ($threadsList ?: '<p>You haven\'t created any threads yet.</p>') . '
                </div>
            </div>
        ';
        
        $this->renderPage('Profile', $content);
    }

    private function processProfileUpdate() {
        if (!$this->currentUser || !$this->settings['enable_signatures']) return;
        
        $signature = trim($_POST['signature'] ?? '');
        if (strlen($signature) > $this->settings['max_signature_length']) {
            $signature = substr($signature, 0, $this->settings['max_signature_length']);
        }
        
        $usersData = json_decode(file_get_contents($this->dataFiles['users']), true);
        foreach ($usersData['users'] as &$user) {
            if ($user['id'] == $this->currentUser['id']) {
                $user['signature'] = $signature;
                break;
            }
        }
        file_put_contents($this->dataFiles['users'], json_encode($usersData, JSON_PRETTY_PRINT));
        
        $this->showSuccess('Signature updated successfully!');
    }

    // PRIVATE MESSAGING
    private function showInbox() {
        $messagesData = json_decode(file_get_contents($this->dataFiles['messages']), true);
        $messages = [];
        
        foreach ($messagesData['messages'] as $msg) {
            if ($msg['to_user_id'] == $this->currentUser['id']) {
                $messages[] = $msg;
            }
        }
        
        usort($messages, function($a, $b) {
            return strtotime($b['sent_at']) - strtotime($a['sent_at']);
        });
        
        $messagesList = '';
        foreach ($messages as $msg) {
            $fromUser = $this->getUserById($msg['from_user_id']);
            $unreadClass = !$msg['is_read'] ? 'style="background:rgba(255,255,0,0.1);"' : '';
            $messagesList .= '
                <div class="post" ' . $unreadClass . '>
                    <div style="display:flex;justify-content:space-between;align-items:center;">
                        <div>
                            <strong>' . (!$msg['is_read'] ? '📩 ' : '') . '<a href="?action=messages&method=view&id=' . $msg['id'] . '">' . htmlspecialchars($msg['subject']) . '</a></strong><br>
                            <small>From: ' . htmlspecialchars($fromUser['username']) . ' • ' . $this->timeAgo($msg['sent_at']) . '</small>
                        </div>
                        <div>
                            <a href="?action=messages&method=delete&process=1&id=' . $msg['id'] . '" class="btn btn-small btn-danger" onclick="return confirm(\'Delete this message?\')">Delete</a>
                        </div>
                    </div>
                </div>
            ';
        }
        
        $content = '
            <div class="admin-panel">
                <h2>✉️ Inbox</h2>
                <div style="margin-bottom:20px;">
                    <a href="?action=messages&method=inbox" class="btn">Inbox</a>
                    <a href="?action=messages&method=sent" class="btn">Sent</a>
                    <a href="?action=messages&method=compose" class="btn btn-success">Compose</a>
                </div>
                ' . ($messagesList ?: '<div class="info">Your inbox is empty.</div>') . '
            </div>
        ';
        
        $this->renderPage('Messages', $content);
    }

    private function showSent() {
        $messagesData = json_decode(file_get_contents($this->dataFiles['messages']), true);
        $messages = [];
        
        foreach ($messagesData['messages'] as $msg) {
            if ($msg['from_user_id'] == $this->currentUser['id']) {
                $messages[] = $msg;
            }
        }
        
        usort($messages, function($a, $b) {
            return strtotime($b['sent_at']) - strtotime($a['sent_at']);
        });
        
        $messagesList = '';
        foreach ($messages as $msg) {
            $toUser = $this->getUserById($msg['to_user_id']);
            $messagesList .= '
                <div class="post">
                    <div style="display:flex;justify-content:space-between;align-items:center;">
                        <div>
                            <strong><a href="?action=messages&method=view&id=' . $msg['id'] . '">' . htmlspecialchars($msg['subject']) . '</a></strong><br>
                            <small>To: ' . htmlspecialchars($toUser['username']) . ' • ' . $this->timeAgo($msg['sent_at']) . '</small>
                        </div>
                    </div>
                </div>
            ';
        }
        
        $content = '
            <div class="admin-panel">
                <h2>📤 Sent Messages</h2>
                <div style="margin-bottom:20px;">
                    <a href="?action=messages&method=inbox" class="btn">Inbox</a>
                    <a href="?action=messages&method=sent" class="btn">Sent</a>
                    <a href="?action=messages&method=compose" class="btn btn-success">Compose</a>
                </div>
                ' . ($messagesList ?: '<div class="info">You haven\'t sent any messages yet.</div>') . '
            </div>
        ';
        
        $this->renderPage('Sent Messages', $content);
    }

    private function showCompose() {
        $toUsername = $_GET['to'] ?? '';
        
        $content = '
            <div class="admin-panel">
                <h2>✍️ Compose Message</h2>
                <form method="POST" action="?action=messages&method=send&process=1">
                    <div class="form-group">
                        <label>To (Username)</label>
                        <input type="text" name="to_username" value="' . htmlspecialchars($toUsername) . '" required placeholder="Enter username...">
                    </div>
                    <div class="form-group">
                        <label>Subject</label>
                        <input type="text" name="subject" required maxlength="100" placeholder="Message subject...">
                    </div>
                    <div class="form-group">
                        <label>Message</label>
                        <textarea name="content" rows="8" required placeholder="Your message..."></textarea>
                    </div>
                    <button type="submit" class="btn">Send Message</button>
                    <a href="?action=messages" class="btn" style="background:#666;">Cancel</a>
                </form>
            </div>
        ';
        
        $this->renderPage('Compose Message', $content);
    }

    private function showMessage() {
        $messageId = (int)($_GET['id'] ?? 0);
        $messagesData = json_decode(file_get_contents($this->dataFiles['messages']), true);
        $message = null;
        
        foreach ($messagesData['messages'] as &$msg) {
            if ($msg['id'] == $messageId) {
                if ($msg['to_user_id'] == $this->currentUser['id'] || $msg['from_user_id'] == $this->currentUser['id']) {
                    $message = &$msg;
                    // Mark as read
                    if ($msg['to_user_id'] == $this->currentUser['id'] && !$msg['is_read']) {
                        $msg['is_read'] = true;
                        file_put_contents($this->dataFiles['messages'], json_encode($messagesData, JSON_PRETTY_PRINT));
                    }
                    break;
                }
            }
        }
        
        if (!$message) {
            $this->showError('Message not found.');
            return;
        }
        
        $fromUser = $this->getUserById($message['from_user_id']);
        $toUser = $this->getUserById($message['to_user_id']);
        
        $content = '
            <div class="admin-panel">
                <h2>' . htmlspecialchars($message['subject']) . '</h2>
                <div class="post">
                    <p><strong>From:</strong> ' . htmlspecialchars($fromUser['username']) . '</p>
                    <p><strong>To:</strong> ' . htmlspecialchars($toUser['username']) . '</p>
                    <p><strong>Sent:</strong> ' . date('M j, Y g:i A', strtotime($message['sent_at'])) . '</p>
                    <hr style="margin:15px 0;opacity:0.3;">
                    <div>' . $this->parseBBCode($message['content']) . '</div>
                </div>
                <div style="margin-top:20px;">
                    <a href="?action=messages&method=compose&to=' . urlencode($fromUser['username']) . '" class="btn">Reply</a>
                    <a href="?action=messages" class="btn" style="background:#666;">Back to Inbox</a>
                    <a href="?action=messages&method=delete&process=1&id=' . $message['id'] . '" class="btn btn-danger" onclick="return confirm(\'Delete this message?\')">Delete</a>
                </div>
            </div>
        ';
        
        $this->renderPage('Message', $content);
    }

    private function processSendMessage() {
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $toUsername = trim($_POST['to_username'] ?? '');
            $subject = trim($_POST['subject'] ?? '');
            $content = trim($_POST['content'] ?? '');
            
            if (empty($toUsername) || empty($subject) || empty($content)) {
                $this->showError('All fields are required.');
                return;
            }
            
            $usersData = json_decode(file_get_contents($this->dataFiles['users']), true);
            $toUser = null;
            
            foreach ($usersData['users'] as $user) {
                if (strtolower($user['username']) === strtolower($toUsername)) {
                    $toUser = $user;
                    break;
                }
            }
            
            if (!$toUser) {
                $this->showError('User not found.');
                return;
            }
            
            $messagesData = json_decode(file_get_contents($this->dataFiles['messages']), true);
            $newMessageId = ++$messagesData['last_id'];
            
            $newMessage = [
                'id' => $newMessageId,
                'from_user_id' => $this->currentUser['id'],
                'to_user_id' => $toUser['id'],
                'subject' => $subject,
                'content' => $content,
                'sent_at' => date('Y-m-d H:i:s'),
                'is_read' => false
            ];
            
            $messagesData['messages'][] = $newMessage;
            file_put_contents($this->dataFiles['messages'], json_encode($messagesData, JSON_PRETTY_PRINT));
            
            header('Location: ?action=messages&method=sent');
            exit;
        }
    }

    private function processDeleteMessage() {
        $messageId = (int)($_GET['id'] ?? 0);
        
        $messagesData = json_decode(file_get_contents($this->dataFiles['messages']), true);
        $messagesData['messages'] = array_filter($messagesData['messages'], function($msg) use ($messageId) {
            if ($msg['id'] == $messageId) {
                return !($msg['to_user_id'] == $this->currentUser['id'] || $msg['from_user_id'] == $this->currentUser['id']);
            }
            return true;
        });
        $messagesData['messages'] = array_values($messagesData['messages']);
        file_put_contents($this->dataFiles['messages'], json_encode($messagesData, JSON_PRETTY_PRINT));
        
        header('Location: ?action=messages');
        exit;
    }

    // ADMIN PANEL
    private function showAdminDashboard() {
        $stats = [
            'users' => $this->getUserCount(),
            'threads' => $this->getThreadCount(),
            'posts' => $this->getPostCount(),
            'categories' => $this->getCategoryCount(),
            'reports' => $this->getPendingReportsCount()
        ];

        $content = '
            <div class="admin-panel">
                <h1>⚙️ Admin Dashboard</h1>
                <p>Welcome, ' . htmlspecialchars($this->currentUser['username']) . '! You are managing the forum.</p>
                
                <div class="stats-grid">
                    <div class="stat-card">
                        <h3>' . $stats['users'] . '</h3>
                        <p>Users</p>
                    </div>
                    <div class="stat-card">
                        <h3>' . $stats['threads'] . '</h3>
                        <p>Threads</p>
                    </div>
                    <div class="stat-card">
                        <h3>' . $stats['posts'] . '</h3>
                        <p>Posts</p>
                    </div>
                    <div class="stat-card">
                        <h3>' . $stats['categories'] . '</h3>
                        <p>Categories</p>
                    </div>
                    <div class="stat-card">
                        <h3>' . $stats['reports'] . '</h3>
                        <p>Pending Reports</p>
                    </div>
                </div>
                
                <div style="display:grid;grid-template-columns:repeat(auto-fit,minmax(250px,1fr));gap:15px;margin-top:30px;">
                    <a href="?action=admin&method=users" class="post">
                        <h3>👥 Users</h3>
                        <p>Manage user accounts, roles, and bans</p>
                    </a>
                    <a href="?action=admin&method=moderators" class="post">
                        <h3>🛡️ Moderators</h3>
                        <p>Promote users to moderator status</p>
                    </a>
                    <a href="?action=admin&method=categories" class="post">
                        <h3>📁 Categories</h3>
                        <p>Organize forum categories</p>
                    </a>
                    <a href="?action=admin&method=themes" class="post">
                        <h3>🎨 Themes</h3>
                        <p>Customize forum appearance</p>
                    </a>
                    <a href="?action=admin&method=settings" class="post">
                        <h3>⚙️ Settings</h3>
                        <p>Configure system settings</p>
                    </a>
                    <a href="?action=admin&method=reports" class="post">
                        <h3>🚨 Reports</h3>
                        <p>Review user reports</p>
                    </a>
                    <a href="?action=admin&method=system" class="post">
                        <h3>💻 System Info</h3>
                        <p>View system information</p>
                    </a>
                </div>
            </div>
        ';
        
        $this->renderPage('Admin Dashboard', $content);
    }

    private function handleUserManagement() {
        if (isset($_GET['ban'])) {
            $userId = (int)$_GET['ban'];
            $this->toggleUserBan($userId);
            header('Location: ?action=admin&method=users');
            exit;
        }
        
        $usersData = json_decode(file_get_contents($this->dataFiles['users']), true);
        $users = $usersData['users'];
        
        $userRows = '';
        foreach ($users as $user) {
            $badge = '';
            if ($user['role'] === 'admin') $badge = '<span class="badge badge-admin">ADMIN</span>';
            elseif ($user['role'] === 'moderator') $badge = '<span class="badge badge-mod">MOD</span>';
            
            $userRows .= '
                <tr>
                    <td>' . $user['id'] . '</td>
                    <td>' . htmlspecialchars($user['username']) . ' ' . $badge . '</td>
                    <td>' . htmlspecialchars($user['email']) . '</td>
                    <td>' . $user['role'] . '</td>
                    <td>' . $user['reputation'] . '</td>
                    <td>' . $user['post_count'] . '</td>
                    <td>' . ($user['is_banned'] ? '🚫 Banned' : '✅ Active') . '</td>
                    <td>
                        ' . ($user['id'] != $this->currentUser['id'] && $user['role'] !== 'admin' ? '
                        <a href="?action=admin&method=users&ban=' . $user['id'] . '" class="btn btn-small ' . ($user['is_banned'] ? 'btn-success' : 'btn-danger') . '" onclick="return confirm(\'Toggle ban status?\')">
                            ' . ($user['is_banned'] ? 'Unban' : 'Ban') . '
                        </a>' : '') . '
                    </td>
                </tr>
            ';
        }
        
        $content = '
            <div class="admin-panel">
                <h2>👥 User Management</h2>
                <p>Total Users: ' . count($users) . '</p>
                <table class="forum-table">
                    <thead>
                        <tr><th>ID</th><th>Username</th><th>Email</th><th>Role</th><th>Rep</th><th>Posts</th><th>Status</th><th>Actions</th></tr>
                    </thead>
                    <tbody>' . $userRows . '</tbody>
                </table>
            </div>
        ';
        
        $this->renderPage('User Management', $content);
    }

    private function handleModeratorManagement() {
        if (isset($_GET['promote'])) {
            $userId = (int)$_GET['promote'];
            $this->promoteToModerator($userId);
            header('Location: ?action=admin&method=moderators');
            exit;
        }
        
        if (isset($_GET['demote'])) {
            $userId = (int)$_GET['demote'];
            $this->demoteFromModerator($userId);
            header('Location: ?action=admin&method=moderators');
            exit;
        }
        
        $usersData = json_decode(file_get_contents($this->dataFiles['users']), true);
        $moderators = [];
        $regularUsers = [];
        
        foreach ($usersData['users'] as $user) {
            if ($user['role'] === 'moderator') {
                $moderators[] = $user;
            } elseif ($user['role'] === 'user') {
                $regularUsers[] = $user;
            }
        }
        
        $modRows = '';
        foreach ($moderators as $mod) {
            $modRows .= '
                <tr>
                    <td>' . htmlspecialchars($mod['username']) . '</td>
                    <td>' . $mod['reputation'] . '</td>
                    <td>' . $mod['post_count'] . '</td>
                    <td>' . date('M j, Y', strtotime($mod['joined_date'])) . '</td>
                    <td>
                        <a href="?action=admin&method=moderators&demote=' . $mod['id'] . '" class="btn btn-small btn-danger" onclick="return confirm(\'Demote this moderator?\')">Demote</a>
                    </td>
                </tr>
            ';
        }
        
        $userRows = '';
        foreach ($regularUsers as $user) {
            if ($user['post_count'] >= 10) {
                $userRows .= '
                    <tr>
                        <td>' . htmlspecialchars($user['username']) . '</td>
                        <td>' . $user['reputation'] . '</td>
                        <td>' . $user['post_count'] . '</td>
                        <td>
                            <a href="?action=admin&method=moderators&promote=' . $user['id'] . '" class="btn btn-small btn-success" onclick="return confirm(\'Promote to moderator?\')">Promote</a>
                        </td>
                    </tr>
                ';
            }
        }
        
        $content = '
            <div class="admin-panel">
                <h2>🛡️ Moderator Management</h2>
                
                <h3>Current Moderators</h3>
                <table class="forum-table">
                    <thead>
                        <tr><th>Username</th><th>Reputation</th><th>Posts</th><th>Joined</th><th>Actions</th></tr>
                    </thead>
                    <tbody>' . ($modRows ?: '<tr><td colspan="5">No moderators yet.</td></tr>') . '</tbody>
                </table>
                
                <h3 style="margin-top:30px;">Promote to Moderator</h3>
                <p>Active users with 10+ posts eligible for promotion:</p>
                <table class="forum-table">
                    <thead>
                        <tr><th>Username</th><th>Reputation</th><th>Posts</th><th>Actions</th></tr>
                    </thead>
                    <tbody>' . ($userRows ?: '<tr><td colspan="4">No eligible users.</td></tr>') . '</tbody>
                </table>
            </div>
        ';
        
        $this->renderPage('Moderator Management', $content);
    }

    private function handleCategoryManagement() {
        $categoriesData = json_decode(file_get_contents($this->dataFiles['categories']), true);
        $categories = $categoriesData['categories'];
        
        usort($categories, function($a, $b) {
            return $a['order'] - $b['order'];
        });
        
        $categoryRows = '';
        foreach ($categories as $category) {
            $threadCount = $this->getThreadCountByCategory($category['id']);
            $categoryRows .= '
                <tr>
                    <td>' . $category['id'] . '</td>
                    <td>' . htmlspecialchars($category['name']) . '</td>
                    <td>' . htmlspecialchars($category['description']) . '</td>
                    <td>' . $threadCount . '</td>
                    <td>' . $category['order'] . '</td>
                </tr>
            ';
        }
        
        $content = '
            <div class="admin-panel">
                <h2>📁 Category Management</h2>
                <table class="forum-table">
                    <thead>
                        <tr><th>ID</th><th>Name</th><th>Description</th><th>Threads</th><th>Order</th></tr>
                    </thead>
                    <tbody>' . $categoryRows . '</tbody>
                </table>
                <p style="margin-top:20px;"><em>Note: Category editing can be done directly in the JSON file at ' . $this->dataFiles['categories'] . '</em></p>
            </div>
        ';
        
        $this->renderPage('Category Management', $content);
    }

    private function showThemeManagement() {
        if (isset($_GET['set'])) {
            $newTheme = $_GET['set'];
            if (array_key_exists($newTheme, $this->themes)) {
                $this->updateSetting('default_theme', $newTheme);
                header('Location: ?action=admin&method=themes');
                exit;
            }
        }
        
        $currentTheme = $this->settings['default_theme'] ?? 'cyber_nexus';
        
        $themeOptions = '';
        foreach ($this->themes as $key => $name) {
            $isCurrent = $currentTheme === $key;
            $themeOptions .= '
                <div class="post" style="text-align:center;">
                    <h3>' . $name . '</h3>
                    ' . ($isCurrent ? 
                        '<div class="btn btn-success" style="margin:10px;">✅ Active</div>' : 
                        '<a href="?action=admin&method=themes&set=' . $key . '" class="btn" style="margin:10px;">Select Theme</a>'
                    ) . '
                </div>
            ';
        }
        
        $content = '
            <div class="admin-panel">
                <h2>🎨 Theme Management</h2>
                <p>Current Theme: <strong>' . $this->themes[$currentTheme] . '</strong></p>
                
                <div style="display:grid;grid-template-columns:repeat(auto-fit,minmax(200px,1fr));gap:15px;margin:20px 0;">
                    ' . $themeOptions . '
                </div>
            </div>
        ';
        
        $this->renderPage('Theme Management', $content);
    }

    private function handleSettingsManagement() {
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $checkboxFields = ['registration_enabled', 'allow_bbcode', 'allow_images', 'enable_reputation', 'enable_signatures', 'enable_private_messages', 'enable_reports', 'maintenance_mode', 'enable_search'];
            
            foreach ($this->settings as $key => $currentValue) {
                if (in_array($key, $checkboxFields)) {
                    $this->updateSetting($key, isset($_POST[$key]) && $_POST[$key] == '1');
                } elseif (isset($_POST[$key])) {
                    $this->updateSetting($key, $_POST[$key]);
                }
            }
            
            $this->showSuccess('Settings updated successfully!');
            return;
        }
        
        $content = '
            <div class="admin-panel">
                <h2>⚙️ System Settings</h2>
                <form method="POST">
                    <div style="display:grid;grid-template-columns:1fr 1fr;gap:20px;">
                        <div class="post">
                            <h3>General Settings</h3>
                            <div class="form-group">
                                <label>Site Title</label>
                                <input type="text" name="site_title" value="' . htmlspecialchars($this->settings['site_title']) . '">
                            </div>
                            <div class="form-group">
                                <label>Site Description</label>
                                <textarea name="site_description" rows="3">' . htmlspecialchars($this->settings['site_description']) . '</textarea>
                            </div>
                            <div class="form-group">
                                <label>Admin Email</label>
                                <input type="email" name="admin_email" value="' . htmlspecialchars($this->settings['admin_email']) . '">
                            </div>
                            <div class="form-group">
                                <label>Meta Description</label>
                                <textarea name="meta_description" rows="2">' . htmlspecialchars($this->settings['meta_description']) . '</textarea>
                            </div>
                            <div class="form-group">
                                <label>Meta Keywords</label>
                                <input type="text" name="meta_keywords" value="' . htmlspecialchars($this->settings['meta_keywords']) . '">
                            </div>
                        </div>
                        
                        <div class="post">
                            <h3>Forum Settings</h3>
                            <div class="form-group">
                                <label>Posts Per Page</label>
                                <input type="number" name="posts_per_page" value="' . $this->settings['posts_per_page'] . '" min="5" max="100">
                            </div>
                            <div class="form-group">
                                <label>Threads Per Page</label>
                                <input type="number" name="threads_per_page" value="' . $this->settings['threads_per_page'] . '" min="5" max="100">
                            </div>
                            <div class="form-group">
                                <label>Max Signature Length</label>
                                <input type="number" name="max_signature_length" value="' . $this->settings['max_signature_length'] . '" min="50" max="500">
                            </div>
                            <div class="form-group">
                                <label>Post Edit Time Limit (seconds)</label>
                                <input type="number" name="post_edit_time_limit" value="' . $this->settings['post_edit_time_limit'] . '" min="60" max="3600">
                                <small style="opacity:0.8;">' . ($this->settings['post_edit_time_limit'] / 60) . ' minutes</small>
                            </div>
                        </div>
                        
                        <div class="post">
                            <h3>Feature Toggles</h3>
                            <div class="form-group">
                                <label><input type="checkbox" name="registration_enabled" value="1" ' . ($this->settings['registration_enabled'] ? 'checked' : '') . '> Allow Registration</label>
                            </div>
                            <div class="form-group">
                                <label><input type="checkbox" name="allow_bbcode" value="1" ' . ($this->settings['allow_bbcode'] ? 'checked' : '') . '> Allow BBCode</label>
                            </div>
                            <div class="form-group">
                                <label><input type="checkbox" name="allow_images" value="1" ' . ($this->settings['allow_images'] ? 'checked' : '') . '> Allow Images</label>
                            </div>
                            <div class="form-group">
                                <label><input type="checkbox" name="enable_reputation" value="1" ' . ($this->settings['enable_reputation'] ? 'checked' : '') . '> Enable Reputation</label>
                            </div>
                            <div class="form-group">
                                <label><input type="checkbox" name="enable_signatures" value="1" ' . ($this->settings['enable_signatures'] ? 'checked' : '') . '> Enable Signatures</label>
                            </div>
                            <div class="form-group">
                                <label><input type="checkbox" name="enable_private_messages" value="1" ' . ($this->settings['enable_private_messages'] ? 'checked' : '') . '> Enable Private Messages</label>
                            </div>
                            <div class="form-group">
                                <label><input type="checkbox" name="enable_reports" value="1" ' . ($this->settings['enable_reports'] ? 'checked' : '') . '> Enable Reports</label>
                            </div>
                            <div class="form-group">
                                <label><input type="checkbox" name="enable_search" value="1" ' . ($this->settings['enable_search'] ? 'checked' : '') . '> Enable Search</label>
                            </div>
                            <div class="form-group">
                                <label><input type="checkbox" name="maintenance_mode" value="1" ' . ($this->settings['maintenance_mode'] ? 'checked' : '') . '> Maintenance Mode</label>
                            </div>
                        </div>
                    </div>
                    
                    <button type="submit" class="btn btn-success" style="padding:15px 40px;font-size:1.1em;margin-top:20px;">💾 Save All Settings</button>
                </form>
            </div>
        ';
        
        $this->renderPage('Settings Management', $content);
    }

    private function showReports() {
        $reportsData = json_decode(file_get_contents($this->dataFiles['reports']), true);
        $reports = $reportsData['reports'] ?? [];
        
        usort($reports, function($a, $b) {
            if ($a['status'] != $b['status']) {
                return $a['status'] === 'pending' ? -1 : 1;
            }
            return strtotime($b['created_at']) - strtotime($a['created_at']);
        });
        
        $reportRows = '';
        foreach ($reports as $report) {
            $post = $this->getPostById($report['post_id']);
            $reporter = $this->getUserById($report['reported_by']);
            $thread = $this->getThreadById($report['thread_id']);
            
            $statusClass = $report['status'] === 'pending' ? 'style="background:rgba(255,255,0,0.1);"' : '';
            
            $reportRows .= '
                <tr ' . $statusClass . '>
                    <td>' . $report['id'] . '</td>
                    <td><a href="?action=forum&method=thread&id=' . $report['thread_id'] . '#post-' . $report['post_id'] . '">Post #' . $report['post_id'] . '</a><br>
                        <small>in "' . htmlspecialchars($thread['title']) . '"</small></td>
                    <td>' . htmlspecialchars($reporter['username']) . '</td>
                    <td>' . htmlspecialchars($report['reason']) . '</td>
                    <td>' . $this->timeAgo($report['created_at']) . '</td>
                    <td>' . ucfirst($report['status']) . '</td>
                </tr>
            ';
        }
        
        $content = '
            <div class="admin-panel">
                <h2>🚨 User Reports</h2>
                <p>Total Reports: ' . count($reports) . ' | Pending: ' . $this->getPendingReportsCount() . '</p>
                <table class="forum-table">
                    <thead>
                        <tr><th>ID</th><th>Post</th><th>Reporter</th><th>Reason</th><th>Date</th><th>Status</th></tr>
                    </thead>
                    <tbody>' . ($reportRows ?: '<tr><td colspan="6">No reports yet.</td></tr>') . '</tbody>
                </table>
            </div>
        ';
        
        $this->renderPage('Reports', $content);
    }

    private function showSystemInfo() {
        $systemInfo = [
            'PHP Version' => PHP_VERSION,
            'Server Software' => $_SERVER['SERVER_SOFTWARE'] ?? 'Unknown',
            'Database Type' => 'JSON Files',
            'Data Directory' => DATA_DIR,
            'Upload Directory' => UPLOAD_DIR,
            'Forum Version' => VERSION
        ];

        $infoHtml = '';
        foreach ($systemInfo as $key => $value) {
            $infoHtml .= '<tr><td><strong>' . $key . '</strong></td><td>' . htmlspecialchars($value) . '</td></tr>';
        }

        $content = '
            <div class="admin-panel">
                <h2>💻 System Information</h2>
                <table class="forum-table">
                    <tbody>' . $infoHtml . '</tbody>
                </table>
            </div>
        ';
        
        $this->renderPage('System Info', $content);
    }

    // UTILITY METHODS
    private function updateSetting($key, $value) {
        $settingsData = json_decode(file_get_contents($this->dataFiles['settings']), true);
        $settingsData['settings'][$key] = $value;
        file_put_contents($this->dataFiles['settings'], json_encode($settingsData, JSON_PRETTY_PRINT));
        $this->settings[$key] = $value;
    }

    private function toggleUserBan($userId) {
        $usersData = json_decode(file_get_contents($this->dataFiles['users']), true);
        foreach ($usersData['users'] as &$user) {
            if ($user['id'] == $userId) {
                $user['is_banned'] = !$user['is_banned'];
                break;
            }
        }
        file_put_contents($this->dataFiles['users'], json_encode($usersData, JSON_PRETTY_PRINT));
    }

    private function promoteToModerator($userId) {
        $usersData = json_decode(file_get_contents($this->dataFiles['users']), true);
        foreach ($usersData['users'] as &$user) {
            if ($user['id'] == $userId && $user['role'] === 'user') {
                $user['role'] = 'moderator';
                break;
            }
        }
        file_put_contents($this->dataFiles['users'], json_encode($usersData, JSON_PRETTY_PRINT));
    }

    private function demoteFromModerator($userId) {
        $usersData = json_decode(file_get_contents($this->dataFiles['users']), true);
        foreach ($usersData['users'] as &$user) {
            if ($user['id'] == $userId && $user['role'] === 'moderator') {
                $user['role'] = 'user';
                break;
            }
        }
        file_put_contents($this->dataFiles['users'], json_encode($usersData, JSON_PRETTY_PRINT));
    }

    private function incrementUserPostCount($userId) {
        $usersData = json_decode(file_get_contents($this->dataFiles['users']), true);
        foreach ($usersData['users'] as &$user) {
            if ($user['id'] == $userId) {
                $user['post_count']++;
                break;
            }
        }
        file_put_contents($this->dataFiles['users'], json_encode($usersData, JSON_PRETTY_PRINT));
    }

    private function getPostById($id) {
        if (!file_exists($this->dataFiles['posts'])) return null;
        $data = json_decode(file_get_contents($this->dataFiles['posts']), true);
        foreach ($data['posts'] as $post) {
            if ($post['id'] == $id) return $post;
        }
        return null;
    }

    private function getUserCount() {
        if (!file_exists($this->dataFiles['users'])) return 0;
        $data = json_decode(file_get_contents($this->dataFiles['users']), true);
        return count($data['users']);
    }

    private function getThreadCount() {
        if (!file_exists($this->dataFiles['threads'])) return 0;
        $data = json_decode(file_get_contents($this->dataFiles['threads']), true);
        return count($data['threads'] ?? []);
    }

    private function getPostCount() {
        if (!file_exists($this->dataFiles['posts'])) return 0;
        $data = json_decode(file_get_contents($this->dataFiles['posts']), true);
        return count($data['posts'] ?? []);
    }

    private function getCategoryCount() {
        if (!file_exists($this->dataFiles['categories'])) return 0;
        $data = json_decode(file_get_contents($this->dataFiles['categories']), true);
        return count($data['categories'] ?? []);
    }

    private function getPendingReportsCount() {
        if (!file_exists($this->dataFiles['reports'])) return 0;
        $data = json_decode(file_get_contents($this->dataFiles['reports']), true);
        $count = 0;
        foreach ($data['reports'] ?? [] as $report) {
            if ($report['status'] === 'pending') $count++;
        }
        return $count;
    }

    private function getUnreadMessageCount() {
        if (!file_exists($this->dataFiles['messages'])) return 0;
        $data = json_decode(file_get_contents($this->dataFiles['messages']), true);
        $count = 0;
        foreach ($data['messages'] ?? [] as $msg) {
            if ($msg['to_user_id'] == $this->currentUser['id'] && !$msg['is_read']) {
                $count++;
            }
        }
        return $count;
    }

    private function getThreadCountByCategory($categoryId) {
        if (!file_exists($this->dataFiles['threads'])) return 0;
        $data = json_decode(file_get_contents($this->dataFiles['threads']), true);
        $count = 0;
        foreach ($data['threads'] ?? [] as $thread) {
            if ($thread['category_id'] == $categoryId) $count++;
        }
        return $count;
    }

    private function getCategoryById($id) {
        if (!file_exists($this->dataFiles['categories'])) return null;
        $data = json_decode(file_get_contents($this->dataFiles['categories']), true);
        foreach ($data['categories'] as $category) {
            if ($category['id'] == $id) return $category;
        }
        return null;
    }

    private function getThreadById($id) {
        if (!file_exists($this->dataFiles['threads'])) return null;
        $data = json_decode(file_get_contents($this->dataFiles['threads']), true);
        foreach ($data['threads'] as $thread) {
            if ($thread['id'] == $id) return $thread;
        }
        return null;
    }

    private function getThreadsByCategory($categoryId) {
        if (!file_exists($this->dataFiles['threads'])) return [];
        $data = json_decode(file_get_contents($this->dataFiles['threads']), true);
        $threads = [];
        foreach ($data['threads'] ?? [] as $thread) {
            if ($thread['category_id'] == $categoryId) $threads[] = $thread;
        }
        return $threads;
    }

    private function getPostsByThread($threadId) {
        if (!file_exists($this->dataFiles['posts'])) return [];
        $data = json_decode(file_get_contents($this->dataFiles['posts']), true);
        $posts = [];
        foreach ($data['posts'] ?? [] as $post) {
            if ($post['thread_id'] == $threadId) $posts[] = $post;
        }
        usort($posts, function($a, $b) {
            return strtotime($a['created_at']) - strtotime($b['created_at']);
        });
        return $posts;
    }

    private function getReplyCount($threadId) {
        if (!file_exists($this->dataFiles['posts'])) return 0;
        $data = json_decode(file_get_contents($this->dataFiles['posts']), true);
        $count = 0;
        foreach ($data['posts'] ?? [] as $post) {
            if ($post['thread_id'] == $threadId) $count++;
        }
        return max(0, $count - 1);
    }

    private function incrementViewCount($threadId) {
        $data = json_decode(file_get_contents($this->dataFiles['threads']), true);
        foreach ($data['threads'] as &$thread) {
            if ($thread['id'] == $threadId) {
                $thread['view_count'] = ($thread['view_count'] ?? 0) + 1;
                break;
            }
        }
        file_put_contents($this->dataFiles['threads'], json_encode($data, JSON_PRETTY_PRINT));
    }

    private function getLatestThreads($limit = 5) {
        if (!file_exists($this->dataFiles['threads'])) return [];
        $data = json_decode(file_get_contents($this->dataFiles['threads']), true);
        $threads = $data['threads'] ?? [];
        usort($threads, function($a, $b) {
            return strtotime($b['created_at']) - strtotime($a['created_at']);
        });
        return array_slice($threads, 0, $limit);
    }

    private function getLatestThreadByCategory($categoryId) {
        $threads = $this->getThreadsByCategory($categoryId);
        if (empty($threads)) return null;
        
        usort($threads, function($a, $b) {
            return strtotime($b['created_at']) - strtotime($a['created_at']);
        });
        
        return $threads[0];
    }

    private function getLatestPostByThread($threadId) {
        $posts = $this->getPostsByThread($threadId);
        if (empty($posts)) return null;
        
        usort($posts, function($a, $b) {
            return strtotime($b['created_at']) - strtotime($a['created_at']);
        });
        
        return $posts[0];
    }

    private function getTopUsers($limit = 5) {
        if (!file_exists($this->dataFiles['users'])) return [];
        $data = json_decode(file_get_contents($this->dataFiles['users']), true);
        $users = $data['users'];
        
        usort($users, function($a, $b) {
            if ($b['reputation'] != $a['reputation']) {
                return $b['reputation'] - $a['reputation'];
            }
            return $b['post_count'] - $a['post_count'];
        });
        
        return array_slice($users, 0, $limit);
    }

    private function getUserThreads($userId, $limit = 5) {
        if (!file_exists($this->dataFiles['threads'])) return [];
        $data = json_decode(file_get_contents($this->dataFiles['threads']), true);
        $userThreads = [];
        
        foreach ($data['threads'] ?? [] as $thread) {
            if ($thread['user_id'] == $userId) {
                $userThreads[] = $thread;
            }
        }
        
        usort($userThreads, function($a, $b) {
            return strtotime($b['created_at']) - strtotime($a['created_at']);
        });
        
        return array_slice($userThreads, 0, $limit);
    }

    private function timeAgo($datetime) {
        $time = time() - strtotime($datetime);
        if ($time < 1) return 'just now';
        if ($time < 60) return $time . 's ago';
        if ($time < 3600) return round($time/60) . 'm ago';
        if ($time < 86400) return round($time/3600) . 'h ago';
        if ($time < 604800) return round($time/86400) . 'd ago';
        if ($time < 2592000) return round($time/604800) . 'w ago';
        if ($time < 31536000) return round($time/2592000) . 'mo ago';
        return round($time/31536000) . 'y ago';
    }

    private function showError($message) {
        $content = '
            <div class="error">
                <strong>Error:</strong> ' . htmlspecialchars($message) . '
            </div>
            <div style="text-align:center;margin-top:20px;">
                <a href="javascript:history.back()" class="btn">Go Back</a>
                <a href="?action=home" class="btn">Home</a>
            </div>
        ';
        $this->renderPage('Error', $content);
        exit;
    }

    private function showSuccess($message) {
        $content = '
            <div class="success">
                <strong>Success!</strong> ' . htmlspecialchars($message) . '
            </div>
            <div style="text-align:center;margin-top:20px;">
                <a href="javascript:history.back()" class="btn">Go Back</a>
                <a href="?action=home" class="btn">Home</a>
            </div>
        ';
        $this->renderPage('Success', $content);
        exit;
    }
}

// Initialize and run the forum
new Forumly();
?>